import { NextRequest, NextResponse } from 'next/server';
import { openDB } from '@/src/lib/db';

export async function GET(
    request: NextRequest,
    { params }: { params: Promise<{ entityType: string; entityId: string }> }
) {
    try {
        const { entityType, entityId } = await params;

        const db = await openDB();

        return new Promise((resolve) => {
            db.all(
                `SELECT * FROM attachments 
         WHERE entity_type = ? AND entity_id = ? 
         ORDER BY created_at DESC`,
                [entityType, parseInt(entityId)],
                (err, rows: unknown[]) => {
                    if (err) {
                        console.error('Error fetching attachments:', err);
                        resolve(NextResponse.json(
                            { success: false, error: 'Failed to fetch attachments' },
                            { status: 500 }
                        ));
                    } else {
                        const attachments = rows.map(row => ({
                            ...row,
                            is_voice_recording: Boolean(row.is_voice_recording)
                        }));
                        resolve(NextResponse.json({ success: true, attachments }));
                    }
                }
            );
        });
    } catch (error) {
        console.error('Error in GET attachments:', error);
        return NextResponse.json(
            { success: false, error: 'Internal server error' },
            { status: 500 }
        );
    }
}

export async function DELETE(
    request: NextRequest
) {
    try {
        const url = new URL(request.url);
        const attachmentId = url.searchParams.get('id');

        if (!attachmentId) {
            return NextResponse.json(
                { success: false, error: 'Attachment ID is required' },
                { status: 400 }
            );
        }

        const db = await openDB();

        return new Promise((resolve) => {
            db.run(
                'DELETE FROM attachments WHERE id = ?',
                [parseInt(attachmentId)],
                function (err) {
                    if (err) {
                        console.error('Error deleting attachment:', err);
                        resolve(NextResponse.json(
                            { success: false, error: 'Failed to delete attachment' },
                            { status: 500 }
                        ));
                    } else {
                        resolve(NextResponse.json({ success: true }));
                    }
                }
            );
        });
    } catch (error) {
        console.error('Error in DELETE attachment:', error);
        return NextResponse.json(
            { success: false, error: 'Internal server error' },
            { status: 500 }
        );
    }
}

